#include <lib3d/model/loader/md2loader.h>

MD2Loader :: MD2Loader() {
}

MD2Loader :: ~MD2Loader() {
}

MD2Loader* MD2Loader :: instance() {
   static MD2Loader *loader = new MD2Loader;
   return loader;
}

AnimatedModel* MD2Loader :: load(char *path,char *strTexture) {
   
   fp = fopen(path,"rb");
   if (fp == NULL)
      return NULL;

   fread(&m_Header,1,sizeof(tMd2Header),fp);
   
   if (m_Header.magic != 844121161 || m_Header.version != 8)
      return NULL;   

   // Create a larger buffer for the frames of animation (not fully used yet)
   unsigned char buffer[MD2_MAX_FRAMESIZE];

   // Here we allocate all of our memory from the header's information
   m_pSkins     = new tMd2Skin [m_Header.numSkins];
   m_pTexCoords = new tMd2TexCoord [m_Header.numTexCoords];
   m_pTriangles = new tMd2Face [m_Header.numTriangles];
   m_pFrames    = new tMd2Frame [m_Header.numFrames];

   // Next, we start reading in the data by seeking to our skin names offset
   fseek(fp, m_Header.offsetSkins, SEEK_SET);
	
   // Depending on the skin count, we read in each skin for this model
   fread(m_pSkins, sizeof(tMd2Skin), m_Header.numSkins, fp);
	
   // Move the file pointer to the position in the file for texture coordinates
   fseek(fp, m_Header.offsetTexCoords, SEEK_SET);
	
   // Read in all the texture coordinates in one fell swoop
   fread(m_pTexCoords, sizeof(tMd2TexCoord), m_Header.numTexCoords, fp);

   // Move the file pointer to the triangles/face data offset
   fseek(fp, m_Header.offsetTriangles, SEEK_SET);
	
   // Read in the face data for each triangle (vertex and texCoord indices)
   fread(m_pTriangles, sizeof(tMd2Face), m_Header.numTriangles, fp);
			
   // Move the file pointer to the vertices (frames)
   fseek(fp, m_Header.offsetFrames, SEEK_SET);


   for (int i=0; i < m_Header.numFrames; i++) {

      // Assign our alias frame to our buffer memory
      tMd2AliasFrame *pFrame = (tMd2AliasFrame *) buffer;

      // Allocate the memory for the first frame of animation's vertices
      m_pFrames[i].pVertices = new tMd2Triangle [m_Header.numVertices];

      // Read in the first frame of animation
      fread(pFrame, 1, m_Header.frameSize, fp);

      // Copy the name of the animation to our frames array
      strcpy(m_pFrames[i].strName, pFrame->name);
			
      // Store off a vertex array pointer to cut down large lines of code
      tMd2Triangle *pVertices = m_pFrames[i].pVertices;

      // Go through all of the number of vertices and assign the scale and translations.
      // Store the vertices in our current frame's vertex list array, while swapping Y and Z.
      // Notice we also negate the Z axis as well to make the swap correctly. (THIS IS WRONG??)
      for (int j=0; j < m_Header.numVertices; j++) {
         pVertices[j].vertex[0] = pFrame->aliasVertices[j].vertex[0] * pFrame->scale[0] + pFrame->translate[0];
         pVertices[j].vertex[2] = (pFrame->aliasVertices[j].vertex[1] * pFrame->scale[1] + pFrame->translate[1]);
         pVertices[j].vertex[1] = pFrame->aliasVertices[j].vertex[2] * pFrame->scale[2] + pFrame->translate[2];
      }
   }

   AnimatedModel *model = new AnimatedModel;
   model->setPath(path);

   buildModelFromMD2Data(model);
   calculateAndStoreAnimations(model);

   if (strTexture && strcmp(strTexture,"")) {

      tMaterial *mat = new tMaterial;
      mat->texture = new Texture(strTexture);
      model->setMaterials(mat,1);   
   
   }
   
   cleanUp();
   fclose(fp);   
   return model;
}

void MD2Loader :: buildModelFromMD2Data(AnimatedModel *model) {

   int j = 0, i = 0;

   // Like the previous function, not a lot was changed her from the last tutorial.
   // Since we are dealing with multiple key frames, we need to pretty much do the 
   // same things for each frame.  This entails adding a for loop to go through
   // every frame.  We use index 'i' into the m_pFrames array to set each key frame.
   // This is one tricky thing in the middle of this for loop though, since we only
   // need to save the face and texture information once, this is only calculated
   // once and stored in the first frame of animation.  We can then reference the 
   // polygon information for every other frame from the first frame.  That way
   // we don't copy the same information for every key frame, which would make a huge
   // memory footprint.

   int meshCount = m_Header.numFrames;
   tMesh *meshes = new tMesh[meshCount];
   model->setMeshes(meshes,meshCount);
   
   // Create our animation list and store it in our model
   //ParseAnimations(pModel);

   // Go through every key frame and store it's vertices info in our pObject list.
   for (i=0; i < meshCount; i++) {
      
      // Create a local object to store the first frame of animation's data
      tMesh *currentFrame = &meshes[i];

      // Assign the vertex, texture coord and face count to our new structure
      currentFrame->vertexCount = m_Header.numVertices;
      currentFrame->faceCount   = m_Header.numTriangles;
      currentFrame->uvCount     = m_Header.numTexCoords;
      currentFrame->normalCount = 0;
      currentFrame->normals     = NULL;
      currentFrame->materialIndex = 0;
      strncpy(currentFrame->name,m_pFrames[i].strName,NAME_LEN);

      // Allocate memory for the vertices, texture coordinates and face data.
      currentFrame->vertices = new tVertex[currentFrame->vertexCount];

      float maxx=-100000,maxy=-100000,maxz=-100000;
      float minx=100000,miny=100000,minz=100000;
      
      // Go through all of the vertices and assign them over to our structure
      for (j=0; j < currentFrame->vertexCount; j++) {
         currentFrame->vertices[j].x = m_pFrames[i].pVertices[j].vertex[0];
         currentFrame->vertices[j].y = m_pFrames[i].pVertices[j].vertex[1];
         currentFrame->vertices[j].z = m_pFrames[i].pVertices[j].vertex[2];

         // BoundingBox calculation for mesh
         if (maxx < meshes[i].vertices[j].x) maxx = meshes[i].vertices[j].x;
         if (maxy < meshes[i].vertices[j].y) maxy = meshes[i].vertices[j].y;
         if (maxz < meshes[i].vertices[j].z) maxz = meshes[i].vertices[j].z;
         if (minx > meshes[i].vertices[j].x) minx = meshes[i].vertices[j].x;
         if (miny > meshes[i].vertices[j].y) miny = meshes[i].vertices[j].y;
         if (minz > meshes[i].vertices[j].z) minz = meshes[i].vertices[j].z;

      }

      float dx = maxx-minx;
      float dy = maxy-miny;
      float dz = maxz-minz;
      meshes[i].boundingBox.set(dx,dy,dz);
      
      float maxkoord = dx;
      if (dy > maxkoord) maxkoord = dy;
      if (dz > maxkoord) maxkoord = dz;
      meshes[i].boundingSphereRad = maxkoord;

      // We can now free the old vertices stored in this frame of animation
      // since we have them stored in the pObject list of our model
      delete [] m_pFrames[i].pVertices;
	
      // Now comes the tricky part, since we only need to store the UV coordinates
      // and face indices once, we only do this on the first frame.  If the
      // current frame is past the first frame, we just add the current frame
      // as it is to our list, then continue to the next frame.  If we are
      // on the first frame, we still need to store the UV and face information.
      // This will only go in the first frame, and will be referenced from the
      // remaining frames when we animate the model.

      if(i == 0) {

         // We will only get here ONE because we just need this information
         // calculated for the first key frame.
			
         // Allocate memory for our UV coordinates and face information
         currentFrame->faces = new tFace[currentFrame->faceCount];
         currentFrame->uvs = new tUV[currentFrame->uvCount];

         // Go through all of the uv coordinates and assign them over to our structure.
         // The UV coordinates are not normal uv coordinates, they have a pixel ratio of
         // 0 to 256.  We want it to be a 0 to 1 ratio, so we divide the u value by the
         // skin width and the v value by the skin height.  This gives us our 0 to 1 ratio.
         // For some reason also, the v coodinate is flipped upside down.  We just subtract
         // the v coordinate from 1 to remedy this problem.
         for(j=0; j < currentFrame->uvCount; j++) {
            currentFrame->uvs[j].u = m_pTexCoords[j].u / float(m_Header.skinWidth);
            currentFrame->uvs[j].v = 1 - m_pTexCoords[j].v / float(m_Header.skinHeight);
         }

         // Go through all of the face data and assign it over to OUR structure
         for(j=0; j < currentFrame->faceCount; j++) {

            // Assign the vertex indices to our face data
            currentFrame->faces[j].v[0] = m_pTriangles[j].vertexIndices[0];
            currentFrame->faces[j].v[1] = m_pTriangles[j].vertexIndices[1];
            currentFrame->faces[j].v[2] = m_pTriangles[j].vertexIndices[2];

            // Assign the texture coord indices to our face data
            currentFrame->faces[j].uv[0] = m_pTriangles[j].textureIndices[0];
            currentFrame->faces[j].uv[1] = m_pTriangles[j].textureIndices[1];
            currentFrame->faces[j].uv[2] = m_pTriangles[j].textureIndices[2];

         }

      }
      else {
      
         currentFrame->faces = meshes[0].faces;
         currentFrame->uvs = meshes[0].uvs;
      
      }

   }/* for */
}

void MD2Loader :: calculateAndStoreAnimations(AnimatedModel *model) {

   /*
    * Animasyonlarin dogru bicimde gruplandirilmasi icin her animasyon
    * grubuna bir isim verilmeli ve devam eden frameler bu isme eklenen
    * 1 artan numaralarla sonlandirilmalidir..
    *
    * ORNEK: run1,run2,run3,stand01,stand02,hebe101,hebe102,hebe202,hebe203 ...
    * 
    */
   
   char str[256];
   char prev[256];
   int  startframe = 0;
   int  prevAnimNum = -999;
   vector<tAnimation*> animVec;
   
   strcpy(prev,m_pFrames[0].strName);
   for (int j=0;j<strlen(prev) && !isdigit(prev[j]);++j) ;
   
   prevAnimNum = atoi(&prev[j]);
   prev[j] = '\0';
   strcpy(str,prev);

   for (int i=1;i<m_Header.numFrames;++i) {
      
      strcpy(str,m_pFrames[i].strName);
      for (int j=0;j<strlen(str) && !isdigit(str[j]);++j);
      int num = atoi(&str[j]);
      str[j] = '\0';

      int r = strcmp(str,prev); 
      if (r != 0 || (r == 0 && num != prevAnimNum+1)) {
      
         tAnimation *anim = new tAnimation;
         strcpy(anim->name,prev);
         anim->firstFrame = startframe;
         anim->lastFrame = i - 1;
         anim->currSourceFrame = anim->firstFrame;
         anim->currDestFrame = anim->firstFrame + 1;
         animVec.push_back(anim);
         
         strcpy(prev,str);
         startframe = i;

      }

      prevAnimNum = num;
   }

   tAnimation *anim = new tAnimation;
   strcpy(anim->name,str);
   anim->firstFrame = startframe;
   anim->lastFrame = m_Header.numFrames - 1;
   anim->currSourceFrame = startframe;
   if (m_Header.numFrames > 1)
      anim->currDestFrame = startframe + 1;
   else
      anim->currDestFrame = 0;
   animVec.push_back(anim);

   tAnimation *anims = new tAnimation[animVec.size()];
   for (i=0;i<animVec.size();++i)
      anims[i] = *animVec[i];

   model->setAnimations(anims,animVec.size());

}

void MD2Loader :: cleanUp() {

   if(m_pSkins)	 delete  []  m_pSkins;       // Free the skins data
   if(m_pTexCoords) delete []  m_pTexCoords;     // Free the texture coord data
   if(m_pTriangles) delete []  m_pTriangles;     // Free the triangle face data
   if(m_pFrames)	 delete  []  m_pFrames;         // Free the frames of animation

}