#ifndef __VECTOR3D__H__
#define __VECTOR3D__H__

#include <stdio.h>
#include <math.h>
#include <float.h>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

class Vector3d {

public:
   
   Vector3d() {};
   Vector3d(float x,float y,float z);
   Vector3d(int x,int y,int z);
   ~Vector3d() {};

   void  normalize();
   void  zero();
   float length() const;
   bool  equals(const Vector3d *v) const;
   void  invert();
   void  set(float x,float y,float z);
   void  set(const Vector3d *v);
   void  scale(const Vector3d *s);
   void  scale(float sx,float sy,float sz);
   float dot(const Vector3d *v);
   void  cross(const Vector3d *v);
   float angle(const Vector3d *v);
   void  add(const Vector3d *v);
   void  add(float x,float y,float z);
   void  sub(const Vector3d *v);
   void  sub(float x,float y,float z);
   void  print(FILE *fp) const;
   Vector3d* clone() const;
   
   static float dot(const Vector3d *v1,const Vector3d *v2);
   static Vector3d cross(const Vector3d *v1,const Vector3d *v2);
   static float angle(const Vector3d *v1,const Vector3d *v2);

   float  x,y,z;

};

// --- Vector3d inline funcs ---

inline Vector3d :: Vector3d(float x,float y,float z) {
   set(x,y,z);
}

inline Vector3d :: Vector3d(int x,int y,int z) {
   set((float)x,(float)y,(float)z);
}

inline void Vector3d :: zero() {
   set(0.0f,0.0f,0.0f);
}

inline void Vector3d :: normalize() {
   float len = length();
   x /= len;
   y /= len;
   z /= len;
}

inline float Vector3d :: length() const {
   return (float) sqrt(x*x+y*y+z*z);
}

inline bool Vector3d :: equals(const Vector3d *v) const {
   if (x == v->x && y == v->y && z == v->z) return true;
   else return false;
}

inline void Vector3d :: invert() {
   x = -x;
   y = -y;
   z = -z;
}

inline void Vector3d :: set(float x,float y,float z) {
   this->x = x;
   this->y = y;
   this->z = z;
}

inline void Vector3d :: set(const Vector3d *v) {
   this->x = v->x;
   this->y = v->y;
   this->z = v->z;
}

inline void Vector3d :: scale(const Vector3d *s) {
   x *= s->x;
   y *= s->y;
   z *= s->z;
}

inline void Vector3d :: scale(float sx,float sy,float sz) {
   x *= sx;
   y *= sy;
   z *= sz;
}

inline float Vector3d :: dot(const Vector3d *v) {
   return (v->x*x + v->y*y + v->z*z);
}

inline void Vector3d :: cross(const Vector3d *v) {
   x = y * v->z - z * v->y;
   y = z * v->x - x * v->z;
   z = x * v->y - y * v->x;
}

inline float Vector3d :: angle(const Vector3d *v) {
   double angle = acos(dot(v)/(length()*v->length()));
   if(_isnan(angle))
      return 0;
   else
      return (float) (angle * 180.0f / M_PI);
}

inline void Vector3d :: add(const Vector3d *vec) {
   x += vec->x;
   y += vec->y;
   z += vec->z;
}
   
inline void Vector3d :: add(float x,float y,float z) {
   this->x += x;
   this->y += y;
   this->z += z;
}
   
inline void Vector3d :: sub(const Vector3d *vec) {
   x -= vec->x;
   y -= vec->y;
   z -= vec->z;
}
   
inline void Vector3d :: sub(float x,float y,float z) {
   this->x -= x;
   this->y -= y;
   this->z -= z;
}

inline Vector3d Vector3d :: cross(const Vector3d *vec1,const Vector3d *vec2) {

   Vector3d vec3;
   vec3.x = vec1->y * vec2->z - vec1->z * vec2->y;
   vec3.y = vec1->z * vec2->x - vec1->x * vec2->z;
   vec3.z = vec1->x * vec2->y - vec1->y * vec2->x;
   return vec3;
}
   
inline float Vector3d :: dot(const Vector3d *v1,const Vector3d *v2) {
   return (v1->x*v2->x + v1->y*v2->y + v1->z*v2->z);
}

inline float Vector3d :: angle(const Vector3d *v1,const Vector3d *v2) {
   return (float) (acos(Vector3d::dot(v1,v2)/(v1->length()*v2->length())) * 180.0f/M_PI);
}

inline void Vector3d :: print(FILE *fp) const {
   fprintf(fp,"[%f,%f,%f]",x,y,z);
}

inline Vector3d* Vector3d :: clone() const {
   return new Vector3d(x,y,z);
}


#endif