/*
 * Lib3d Ornekleri -- 2
 *
 *
 * Konu:
 *    Detayli arazi goruntuleme, model animasyonlari,
 *    shader kullanimi, 3rd person kamera acisi kullanimi,
 *    frame rate den bagimsiz calisma
 *
 * Kullanilan Siniflar:
 *    Terrain, AnimatedModel, SimpleTimer, Camera, Position3d, Texture
 *
 * Not:
 *    Lib3d de bulunan bazi fonksiyon eksikliklerinden dolayi ornek
 *    icerisinde model animasyonu ile ilgili bazi bug lar bulunmaktadir..
 *    Lib3d nin yeni versiyonunda bu eksiklikler duzeltilecektir.
 *
 *    :: Deniz Aydinoglu    :: 2002 ::
 *    :: www.oyunyapimi.org :: 2002 ::
 */

#include "init.h"
#include <lib3d/lib3d.h>
#include <lib3d/util/SimpleTimer.h>
#include <lib3d/terrain/Terrain.h>
#include <lib3d/util3d/Camera.h>
#include <lib3d/util3d/Vector3d.h>
#include <lib3d/util3d/Position3d.h>
#include <lib3d/model/AnimatedModel.h>
#include <lib3d/model/Model.h>
#include <lib3d/model/loader/MD2Loader.h>

#include <stdlib.h>
#include <time.h>


// Gokyuzu texture i sky.shader isimli shader scripti icinde belirtiliyor.
// Bu script ile gokyuzunde hareketli bir goruntu elde ediliyor.
Texture       tex("sky.shader");
GLUquadricObj *quadratic =  NULL;

// bu degiskenler kesiksiz hareketin gerceklestirilmesi icin kullaniliyor..
bool          key_up=false,key_down=false,key_left=false,key_right=false;

// ana arazi nesnesi
Terrain       *terrain = NULL;

// animasyonlu model nesnesi
AnimatedModel *model    = NULL;

// ana kamera
Camera        cam;

// model pozisyonu
Position3d      pos(new Vector3d(0,0,0),0,0,0);

// bilgisayar hizindan bagimsiz olarak (frame rate independent) hareketlerin
// gerceklestirilmesi isin kullanilan timer nesnesi
SimpleTimer   timer;

// model icerisindeki animasyonlar arasinda gecis yapmamizi saglayan degisken
int           frameNum = 0;


/*
 * Bu yordamda frame rate den bagimsiz bir bicimde karakterimizin
 * pozisyonu degistiriliyor (kesiksiz hareket).
 * dt degiskeni son cagirimdan beri gecen zamani veriyor..
 *
 * Yordam duzgun islemesi icin renderScene() icerisinden bir defa
 * cagirilmali.
 *
 */

void moveActor(long dt) {

   if (key_up) {
      pos.moveForward(0.03 * dt);
   }
   else 
   if (key_down) {
      pos.moveBackward(0.03 * dt);
   }

   if (key_left) {
      pos.turnLeft(0.05 * dt);
   }
   else
   if (key_right) {
      pos.turnRight(0.05 * dt);
   }

}

/*
 * Bu yordam init.cpp deki SDL dongusunde devamli olarak cagiriliyor.
 * Icerisinde opengl cizim islemleri ve oyun mantigi geregi olan devamli
 * guncellemeler yapilabilir.
 *
 * Frame rate den bagimsiz islem yapilabilmesi icin tum hareketlenmelerin
 * (translate, rotate.. gibi) timer dan elde edilen elapsedTime() degeriyle
 * carpilarak uygulanmasina dikkat edilmeli.
 * Ayni zamanda renderScene() yordaminin son satirinda timer.update() birkez
 * cagirilmalidir.
 *
 */

void renderScene() {

   // bu yapinin kullanimi arazi uzerindeki yuksekligin hesaplanmasinda
   // bize hiz kazandirir.
   static PlaneEqCache cache;

   // en son cizim isleminden beri gecen zamani hesapliyoruz.. Bu degiskeni
   // transformasyon parametrelerinde carpan olarak kullnarak frame rate
   // den bagimsiz bir hareket elde edebiliriz...
   long dt = timer.timeElapsed();
   
   // bu mekanizmanin duzgun calisabilmesi icin her cizim dongusu basinda birkez
   // timer.update() yordami cagirilmalidir..
   // ORNEK-1 icerisinde bu kisimde bir bug vardi!!! Bu ornekte o hatayi duzeltiyoruz.
   timer.update();
   
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   glLoadIdentity();

   // kamera bakis acisi ve konum bilgisine gore openGL matrisi
   // ayarlaniyor.. Bu satirlar ile kameramizdan bakmis oluyoruz..
   glRotatef(cam.xangle,1.0f,0.0f,0.0f);
   glRotatef((float)360.0-cam.yangle,0.0f,1.0f,0.0f);
   glTranslatef(-cam.pos.x,-cam.pos.y,-cam.pos.z);
   
   // gokyuzu ciziliyor. Burada gokyuzu icin scroll eden bir shader
   // scripti kullandik. te.begin(), tex.end() satirlari shader in
   // duzgun calismasi icin gereklidir.
   tex.begin();
   glDisable(GL_CULL_FACE);
   glPushMatrix();
   glRotatef(90,1,0,0);
   gluSphere(quadratic,1000,20,20);
   glPopMatrix();
   glEnable(GL_CULL_FACE);
   tex.end();
   
   // arazi goruntuleniyor
   terrain->render();
   
   // Arazi uzerinde modelin bulundugu konuma gore yukseklik bilgisi
   // elde ediliyor. Cache hiz kazanmak icin kullanilmis(kullanilmasa da olurdu)
   // Daha sonra modelin konumu ayarlaniyor (arazi uzerine aliniyor), ve
   // bir miktar buyuklugu degistiriliyor. (devasa gozukmemesi icin), en sonun
   // da da goruntuleniyor (render)
   pos.pos.y = terrain->getHeightOnTerrain(&pos.pos,&cache);
   Vector3d bb = model->getAnimBoundingBox();
   glPushMatrix();
   glTranslatef(pos.pos.x,pos.pos.y,pos.pos.z);
   glScalef(0.1,0.1,0.1);
   glTranslatef(0,bb.y/2,0);
   glRotatef(pos.yangle + 90,0,1,0);
   model->render();
   glPopMatrix();

   // model animasyonu burada saglaniyor (Frame rate den bagimsiz olarak)
   model->animate(dt);
   moveActor(dt);

   // kamera modelin hemen arkasinada ve biraz ustten bakacak sekilde ayarlaniyor
   // boylece 3rd person goruntu elde edilmis oluyor.
   cam.set(&pos);
   cam.moveUp(10);
   cam.turnDown(10);
   cam.moveBackward(20);

   // OpenGL de double buffering gerceklestiriliyor..
   SDL_GL_SwapBuffers();
     
}


/*
 * Bu yordam SDL tarfindan her tusa basilma durumunda cagirilir.
 * sym parametresi basilan tusun kodunu tasimaktadir.
 *
 */
void keyPressed(SDLKey sym) {

   switch (sym) {

      case SDLK_UP:
         key_up   = true;
         key_down = false;
      
         // kosma animasyonunu aktif animasyon haline getiriyoruz
         frameNum = 1;
         model->activeAnimation(frameNum);
         break;

      case SDLK_DOWN:
         key_up   = false;
         key_down = true;
         
         // kosma animasyonunu aktif animasyon haline getiriyoruz
         frameNum = 1;
         model->activeAnimation(frameNum);
         break;

      case SDLK_LEFT:
         key_right = false;
         key_left  = true;
         break;

      case SDLK_RIGHT:
         key_right = true;
         key_left  = false;
         break;

      case SDLK_SPACE:
         // ates etme animasyonu aktif animasyon oluyor.
         frameNum = 2;
         model->activeAnimation(frameNum);
         break;
   }

}

/*
 * Bu yordam SDL tarfindan tus birakilma durumunda cagirilir.
 * sym parametresi birakilan tusun kodunu tasimaktadir.
 *
 */
void keyReleased(SDLKey sym) {

   switch (sym) {

      case SDLK_UP:
         key_up = false;
         // kosma animasyonundan yerinde durma animasyonuna geciliyor.
         frameNum = 0;
         model->activeAnimation(frameNum);
         break;

      case SDLK_DOWN:
         key_down = false;
         // kosma animasyonundan yerinde durma animasyonuna geciliyor.
         frameNum = 0;
         model->activeAnimation(frameNum);
         break;

      case SDLK_LEFT:
         key_left = false;
         break;

      case SDLK_RIGHT:
         key_right = false;
         break;

      case SDLK_SPACE:
         break;

   }

}

/*
 * Programin basinda bir kez otomatik olarak cagirilan rutin.
 *
 */

void initApp() {

   // terrain generator her seferinde farkli arazi uretsin...
   srand(time(NULL));
   
   // 62*62 boyutlarinda bir arazi uretiliyor... 2^n arazilerde bir bug
   // oldugu icin bu boyuttaki arazileri uretmedik.. (64,128,256...)
   // Ayrica arazi biraz buyutuluyor( setScale(...)) , ve shader scripti
   // ile ust katmani belirtiliyor. Script icerisinde normal texture
   // ustunde detay texture i kullanarak daha gercekci bir gorunum elde ediyoruz..
   // Son olarak build() yordamini cagirarak araziyi olusturuyoruz..
   terrain = new Terrain(62);
   terrain->setTexture("terrain.shader");
   terrain->generate(4,2,rand());
   terrain->setScale(20,2,20);
   terrain->build();

   // Bu kisimda diskten bir quake 2 modeli yukluyoruz. Kaplama texture larini
   // ise ilgili shader scripti ile belirtiyoruz. Burada texture mekanizmasinda
   // yer alan bir bug yuzunden direk texture adi yerine shader scripti kullanildi
   // script aslinda sadece kaplama dokusunu belirtiyor.. animasyon ozelligi yok..)
   MD2Loader loader;
   model = loader.load("hobgoblin_small.md2","hobgoblin.shader");
   model->build();
   model->activeAnimation(frameNum);
   model->setSpeed(0.005f);
   model->setLooping(true);

   // gokyuzu icin kullanacagimiz kureyi olusturuyoruz..
   quadratic = gluNewQuadric();
   gluQuadricNormals(quadratic,GLU_SMOOTH);
   gluQuadricTexture(quadratic,GL_TRUE);

   // gokyuzu texture ini (baska bir shader) olusturuyoruz..
   tex.build();
}

/*
 * Program biterken birkez cagirilan rutin..
 *
 */

void endApp() {

   if (model)     delete model;
   if (terrain)   delete terrain;
   if (quadratic) gluDeleteQuadric(quadratic);

}
